﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Windows.Forms;

namespace OtxMappingSample
{
    [OtxScreenSignature]
    public class Screen1
    {
        #region fields

        private string inputText;
        private bool checkedBox = false;
        private static int count = 0;
        private string buttonText;
        private Screen1Form displayWindow;
        private MyStructure myStructure = new MyStructure();
        private MainColors mainColor = MainColors.None;

        public delegate void ScreenSignatureParameterValueChangedEventHandler(string propertyName, object value);

        #endregion

        #region ctors

        public Screen1()
        {
        }

        #endregion

        #region screen methods

        [OtxScreenSignatureParameterValueChangedEvent]
        public event ScreenSignatureParameterValueChangedEventHandler ScreenSignatureParameterValueChangedEvent;

        [OtxOpenScreen]
        public Form OpenScreen()
        {
            if (displayWindow == null)
            {
                displayWindow = new Screen1Form();

                displayWindow.Closed += DiplayWindow_Closed;
                displayWindow.textBoxTest.TextChanged += new EventHandler(textBoxText_TextChanged);
                displayWindow.checkBoxTest.CheckedChanged += new EventHandler(checkBox_CheckedChanged);
                displayWindow.buttonClick.Click += new EventHandler(buttonClick_Click);
                displayWindow.buttonClose.Click += new EventHandler(buttonClose_Click);
                displayWindow.comboBoxEnum.SelectedIndexChanged += ComboBoxEnum_SelectedIndexChanged;
                displayWindow.buttonChangeStructure.Click += ButtonChangeStructure_Click;
            }

            return displayWindow;
        }

        [OtxCloseScreen]
        public void CloseScreen()
        {
            if (displayWindow != null)
            {
                displayWindow.Closed -= DiplayWindow_Closed;
                displayWindow.Close();
                displayWindow = null;
            }
        }

        [OtxHighlightScreen]
        public void HighlightScreen()
        {
            if (displayWindow != null)
            {
                displayWindow.Activate();
            }
        }

        [OtxScreenIsOpen]
        public bool ScreenIsOpen
        {
            get { return displayWindow != null; }
        }

        #endregion

        #region screen parameters

        [OtxScreenSignatureInParameter]
        [Description("The screen title")]
        public string Title
        {
            set
            {
                if (displayWindow == null)
                {
                    return;
                }

                displayWindow.labelTitle.Text = value;
            }
        }

        [OtxScreenSignatureInOutParameter]
        [Description("The input text")]
        public string Text
        {
            get
            {
                return this.inputText;
            }

            set
            {
                this.inputText = value;
                this.displayWindow.textBoxTest.Text = value;
            }
        }

        [OtxScreenSignatureOutParameter]
        [Description("The input text")]
        public bool Checked
        {
            get
            {
                return this.checkedBox;
            }
        }

        [OtxScreenSignatureOutParameter]
        [Description("The input text")]
        public string ButtonText
        {
            get
            {
                return this.buttonText;
            }
        }

        [OtxScreenSignatureInOutParameter]
        [Description("Also OTX-Structure can be mapped")]
        public MyStructure MyStructure
        {
            get
            {
                this.myStructure.IntegerElement++;
                return this.myStructure;
            }

            set
            {
                this.myStructure = value;
                displayWindow.textBoxStructure.Text = "StringElement1 = " + value.StringElement.ToString() + "\r\nIntegerElement = " + value.IntegerElement.ToString() + "\r\nByteFieldElement = " + ByteArrayToString(value.ByteFieldElement) + "\r\nListElement = " + ListToString(value.ListElement) + "\r\nMapElement = " + MapToString(value.MapElement) + "\r\nEnumElement = " + value.EnumElement.ToString();
            }
        }

        [OtxScreenSignatureInOutParameter]
        [Description("Also OTX-Enumeration can be mapped")]
        public MainColors MainColor
        {
            get
            {
                return this.mainColor;
            }

            set
            {
                this.mainColor = value;
                displayWindow.comboBoxEnum.Text = value.ToString();
            }
        }

        #endregion

        #region helper

        private string ByteArrayToString(byte[] byteArray)
        {
            return "0x" + BitConverter.ToString(byteArray).Replace("-", "");
        }

        private string ListToString(List<int> list)
        {
            return "{ " + String.Join<int>(", ", list) + " }";
        }

        private string MapToString(Dictionary<string, string> map)
        {
            return "{ " + String.Join(", ", map.Select(x => x.Key + " : " + x.Value).ToArray()) + " }";
        }

        private void buttonClose_Click(object sender, EventArgs e)
        {
            this.CloseScreen();
        }

        private void buttonClick_Click(object sender, EventArgs e)
        {
            count++;
            this.buttonText = displayWindow.buttonClick.Text + " " + count;
            if (ScreenSignatureParameterValueChangedEvent != null)
            {
                ScreenSignatureParameterValueChangedEvent("ButtonText", this.buttonText);
            }
        }

        private void checkBox_CheckedChanged(object sender, EventArgs e)
        {
            this.checkedBox = displayWindow.checkBoxTest.Checked;
            if (ScreenSignatureParameterValueChangedEvent != null)
            {
                ScreenSignatureParameterValueChangedEvent("Checked", this.checkedBox);
            }
        }

        private void textBoxText_TextChanged(object sender, EventArgs e)
        {
            this.inputText = displayWindow.textBoxTest.Text;
            if (ScreenSignatureParameterValueChangedEvent != null)
            {
                ScreenSignatureParameterValueChangedEvent("Text", this.inputText);
            }
        }

        private void ButtonChangeStructure_Click(object sender, EventArgs e)
        {
            if (ScreenSignatureParameterValueChangedEvent != null)
            {
                //ScreenSignatureParameterValueChangedEvent("MyStructure", this.myStructure);
                ScreenSignatureParameterValueChangedEvent("MyStructure", this.MyStructure);
            }
        }

        private void ComboBoxEnum_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!Enum.TryParse(displayWindow.comboBoxEnum.Text, true, out this.mainColor))
            {
                this.mainColor = MainColors.None;
            }

            if (ScreenSignatureParameterValueChangedEvent != null)
            {
                ScreenSignatureParameterValueChangedEvent("MainColor", this.mainColor);
            }
        }

        private void DiplayWindow_Closed(object sender, EventArgs e)
        {
            CloseScreen();
        }

        #endregion
    }
}
