﻿using Emotive.ViewModel;
using OtxScreens.Attribute;
using System.Collections.Generic;
using System.ComponentModel;
using System.Windows.Controls;

namespace Emotive.Screens
{
    public abstract class ScreenBasis<TUserControl, TViewModel>
        where TUserControl : ContentControl, new()
        where TViewModel : ScreenBasisViewModel, new()
    {
        protected TUserControl contentControl;
        protected TViewModel viewModel;
        protected bool isPropertyChangingByOtx = false;
        public delegate void ScreenSignatureParameterValueChangedEventHandler(string propertyName, object value);

        [OtxScreenSignatureParameterValueChangedEvent]
        public event ScreenSignatureParameterValueChangedEventHandler ScreenSignatureParameterValueChangedEvent;

        public ScreenBasis()
        {
            viewModel = new TViewModel();
            viewModel.PropertyChanged += ViewModel_PropertyChanged;
        }

        [OtxScreenSignatureInParameter]
        [Description("Header of the screen.")]
        public string Header
        {
            set
            {
                viewModel.Header = value;
            }
        }

        [OtxScreenSignatureInParameter]
        [Description("Button F1 is visible or not")]
        public bool F1_Visible
        {
            set
            {
                viewModel.F1_Visible = value;
            }
        }

        [OtxScreenSignatureInParameter]
        [Description("Button F2 is visible or not")]
        public bool F2_Visible
        {
            set
            {
                viewModel.F2_Visible = value;
            }
        }

        [OtxScreenSignatureOutParameter]
        [Description("Returns the name of the last pressed button (e.g., F1 or F2) or the button index.")]
        public string ButtonPressed
        {
            get; private set;
        }

        [OtxScreenSignatureOutParameter]
        [Description("Returns the name of the last pressed key.")]
        public string KeyPressed
        {
            get; private set;
        }

        [OtxScreenSignatureInParameter]
        [Description("Displays the button with the given names.")]
        public List<string> ButtonBarButtons
        {
            private get
            {
                return viewModel.ButtonBarButtons;
            }

            set
            {
                viewModel.ButtonBarButtons = value;

                ButtonBarButtonsPressed = new List<bool>();

                value?.ForEach(x => ButtonBarButtonsPressed.Add(false));
            }
        }

        [OtxScreenSignatureInParameter]
        [Description("Defines which key is assigned to the related button in the button bar.")]
        public List<string> ButtonBarKeyMapping
        {
            set { viewModel.ButtonBarKeyMapping = value; }
        }

        [OtxScreenSignatureOutParameter]
        [Description("If a button of the button bar is pressed, then the list entry that has the same index as the pressed button becomes true.")]
        public List<bool> ButtonBarButtonsPressed { get; private set; }

        private bool darkTheme = false;
        [OtxScreenSignatureInParameter]
        [Description("Switch screen to dark theme.")]
        public bool DarkTheme {
            set
            {
                darkTheme = value;
                viewModel.UpdateTheme(darkTheme);
            }
        }

        [OtxScreenIsOpen]
        public bool ScreenIsOpen
        {
            get { return contentControl != null; }
        }

        [OtxOpenScreen]
        public virtual TUserControl OpenScreen()
        {
            if (contentControl == null)
            {
                contentControl = new TUserControl();
                viewModel.ButtonClicked += ViewModel_ButtonClicked;
                viewModel.KeyPressed += ViewModel_KeyPressed;
                contentControl.DataContext = viewModel;
            }
            return contentControl;
        }

        [OtxHighlightScreen]
        public virtual void HighlightScreen()
        { }

        [OtxCloseScreen]
        public virtual void CloseScreen()
        {
            if (contentControl != null && viewModel != null)
            {
                viewModel.ButtonClicked -= ViewModel_ButtonClicked;
                viewModel.KeyPressed -= ViewModel_KeyPressed;
                viewModel.PropertyChanged -= ViewModel_PropertyChanged;
                contentControl = null;
            }
        }

        protected virtual void ViewModel_PropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e) { }

        private void ViewModel_ButtonClicked(object sender, EventArgs.ButtonClickedEventArgs e)
        {
            ButtonPressed = e.ButtonName;

            InvokeScreenParamChanged(nameof(ButtonPressed), ButtonPressed);

            int indexButton;
            if (int.TryParse(ButtonPressed, out indexButton))
            {
                indexButton -= 1;
            }
            else
            {
                indexButton = ButtonBarButtons?.IndexOf(ButtonPressed) ?? -1;
            }

            if ((indexButton >= 0) && (indexButton < ButtonBarButtonsPressed.Count))
            {
                ButtonBarButtonsPressed[indexButton] = true;

                InvokeScreenParamChanged(nameof(ButtonBarButtonsPressed), new List<bool>(ButtonBarButtonsPressed));
            }
        }

        private void ViewModel_KeyPressed(object sender, System.Windows.Input.KeyEventArgs e)
        {
            KeyPressed = e.Key.ToString();
            InvokeScreenParamChanged(nameof(KeyPressed), KeyPressed);
        }

        protected void InvokeScreenParamChanged(string name, object value)
        {
            ScreenSignatureParameterValueChangedEvent?.Invoke(name, value);
        }
    }
}
