﻿using Emotive.Common;
using Emotive.EventArgs;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Input;

namespace Emotive.ViewModel
{
    public class ScreenBasisViewModel : ViewModelBase
    {
        internal event EventHandler<ButtonClickedEventArgs> ButtonClicked;
        internal event EventHandler<KeyEventArgs> KeyPressed;
        public ScreenBasisViewModel()
        {
            _buttonViewModels = new ObservableCollection<ButtonViewModel>
            {
                new ButtonViewModel(Constants.ButtonNameF1, "F1: Ok")
                {
                    IsVisible = false
                },
                new ButtonViewModel(Constants.ButtonNameF2, "F2: Abbruch")
                {
                    IsVisible = false
                }
            };
        }
        private string header;
        public string Header
        {
            get => header;
            set
            {
                header = value;
                this.OnPropertyChanged(nameof(Header));
            }
        }

        private bool f1_Visible;
        public bool F1_Visible
        {
            get => f1_Visible;
            set
            {
                SetIsVisible(Constants.ButtonNameF1, value);
                f1_Visible = value;
                this.OnPropertyChanged(nameof(F1_Visible));
            }
        }

        private bool f2_Visible;
        public bool F2_Visible
        {
            get => f2_Visible;
            set
            {
                SetIsVisible(Constants.ButtonNameF2, value);
                f2_Visible = value;
                this.OnPropertyChanged(nameof(F2_Visible));
            }
        }

        private void SetIsVisible(string id, bool isVisible)
        {
            var button = _buttonViewModels.FirstOrDefault(x => x.Id == id);

            if (button != null)
            {
                button.IsVisible = isVisible;
            }
        }

        private readonly ObservableCollection<ButtonViewModel> _buttonViewModels;
        public IEnumerable<ButtonViewModel> ButtonViewModels => _buttonViewModels;
        public bool IsButtonBarVisible => _buttonViewModels.Any(x => x.IsVisible);

        private ICommand pressButtonCommand;
        public ICommand PressButtonCommand
        {
            get
            {
                return pressButtonCommand ?? (pressButtonCommand = new CommandHandler(PressButton));
            }
        }

        private ICommand _keyPressedCommand;
        public ICommand KeyPressedCommand => _keyPressedCommand ?? (_keyPressedCommand = new CommandHandler(OnKeyPressed));

        private void OnKeyPressed(object obj)
        {
            KeyPressed?.Invoke(this, obj as KeyEventArgs);
        }

        public void PressButton(object id)
        {
            if (!string.IsNullOrWhiteSpace(id.ToString()) && IsButtonVisible(id.ToString()))
            {
                ButtonClicked?.Invoke(this, new ButtonClickedEventArgs(id.ToString()));
            }
        }

        private bool IsButtonVisible(string id)
        {
            return _buttonViewModels.FirstOrDefault(x => x.Id == id)?.IsVisible ?? false;
        }

        private List<string> buttonBarButtons;
        public List<string> ButtonBarButtons
        {
            get { return buttonBarButtons; }
            set
            {
                buttonBarButtons = value;
                CreateButtonViewModels();
            }
        }

        private List<string> buttonBarKeyMapping;
        public List<string> ButtonBarKeyMapping
        {
            get => buttonBarKeyMapping;
            set
            {
                buttonBarKeyMapping = value;
                CreateButtonViewModels();
            }
        }

        private string textColor = OtxColor.Text.ToString();
        public string TextColor
        {
            get => textColor;
            set
            {
                textColor = value;
                OnPropertyChanged(nameof(TextColor));
            }
        }

        private string backgroundColor = OtxColor.Canvas.ToString();
        public string BackgroundColor
        {
            get => backgroundColor;
            set
            {
                backgroundColor = value;
                OnPropertyChanged(nameof(BackgroundColor));
            }
        }

        private void CreateButtonViewModels()
        {
            _buttonViewModels.Clear();

            for (var n = 0; n < ButtonBarButtons.Count; ++n)
            {
                _buttonViewModels.Add(new ButtonViewModel(ButtonBarKeyMapping?.Count >= (n + 1) ? ButtonBarKeyMapping[n] : (n + 1).ToString(), ButtonBarButtons[n]));
            }

            this.OnPropertyChanged(nameof(IsButtonBarVisible));
        }

        public virtual void UpdateTheme(bool darkTheme)
        {
            CommonColor.Instance.DarkTheme = darkTheme;
            /*BackgroundColor = darkTheme ? OtxColor.Black.ToString() : OtxColor.White.ToString();
            TextColor = darkTheme ? OtxColor.White.ToString() : OtxColor.Black.ToString();*/
        }
    }

    internal class CommandHandler : ICommand
    {
        private Action<object> _action;
        private Func<object, bool> _canExecute;

        /// <summary>
        /// Creates instance of the command handler
        /// </summary>
        /// <param name="action">Action to be executed by the command</param>
        /// <param name="canExecute">A bolean property to containing current permissions to execute the command</param>
        public CommandHandler(Action<object> action, Func<object, bool> canExecute)
        {
            if (action == null) throw new ArgumentNullException(nameof(action));
            _action = action;
            _canExecute = canExecute ?? (x => true);
        }
        public CommandHandler(Action<object> action) : this(action, null)
        {
        }

        /// <summary>
        /// Wires CanExecuteChanged event 
        /// </summary>
        public event EventHandler CanExecuteChanged
        {
            add { CommandManager.RequerySuggested += value; }
            remove { CommandManager.RequerySuggested -= value; }
        }

        /// <summary>
        /// Forcess checking if execute is allowed
        /// </summary>
        /// <param name="parameter"></param>
        /// <returns></returns>
        public bool CanExecute(object parameter)
        {
            return _canExecute(parameter);
        }

        public void Execute(object parameter)
        {
            _action(parameter);
        }
        public void Refresh()
        {
            CommandManager.InvalidateRequerySuggested();
        }
    }
}
